/**
* Copyright (C) 2011-2012, Crestron Electronics, Inc.  All rights reserved.
* No part of this software may be reproduced in any form, machine or natural,
* without the express written consent of Crestron.
*
* \file     (See the name of this file).
* \brief    IR-based dock device driver.
* \detail   See Microsoft Visual C++ Help on the stream driver interface and
*           the functions named as 'XXX_...()'.  Those help topics explain the
*           purpose, parameters, and return-values of functions in this file.
* \warning  (None).
* \note     (None).
* \author   Dennis Griffin, Larry Salant, Tom Yong, Arto Kiremitdjian.
* \date     11/14/2011
* \review   (Pending review).
**/
//-------------------------------------------------------------------------
// Include files.

#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/fsl_devices.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <linux/proc_fs.h>

#include <linux/interrupt.h>
//#include <linux/irq.h>

#include <linux/poll.h>
#include <linux/delay.h>

#include "IrdHwSignals.h"
#include "IrdI2cIF.h" // IrdI2cInterface.

//-------------------------------------------------------------------------
// Preprocessor defines.
#define IRD_TRACE 	0

//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Short name to identify this device/driver in error/log messages.
//
//  - May match the device-name prefix, which would be the value of "Prefix"
//    under a Registry subkey "HKEY_LOCAL_MACHINE/Drivers/BuiltIn/...".
//
#define DEVICE_SHORT_NAME "IRD"
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// Write time-out for the queue of Cresnet packets from the IR transceiver.
#define IRD_MSGQ_WRITE_TIMEOUT (20)

//-------------------------------------------------------------------------
// Type definitions.

#define BUFFER_LEN_MAX		4096

//-------------------------------------------------------------------------
// Global function declarations.

//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Stream driver interface functions that the OS would invoke.
//
//  - If the source code of this driver/DLL is associated with a ".def" file,
//    then the functions declared below should match ones named in that file.
//
//  - Microsoft Visual C++ Help topics about the stream driver interface
//    identify such functions as 'XXX_...()' (for example, 'XXX_Init()').
//

//    unsigned int IRD_Open (unsigned int hDeviceContext, unsigned int AccessCode, unsigned int ShareMode);
//    int  IRD_Close (unsigned int hOpenContext);
//    unsigned int IRD_Read (unsigned int hOpenContext, void* pBuffer, unsigned int Count);
//    unsigned int IRD_Write (unsigned int hOpenContext, LPCVOID pBuffer, unsigned int Count);

//-------------------------------------------------------------------------
// Global data/variable definitions.

DECLARE_COMPLETION(wakeUp);

struct IRD_chip *gChip;
static struct workqueue_struct *ird_interrupt_wq;
static struct workqueue_struct *ird_wakeup_wq;


//-------------------------------------------------------------------------
// Local data/variables (declarations and definitions).

static void *ird_buf;
static unsigned int ird_buf_len = 0;

// Synchronizes operations that affect shared/common resources.
//  - Protects all or most of the local objects defined further below.
//  - Protects some of the local functions declared further below.

// State variable: Whether the device is detecting/receiving an IR signal,
//  presumably from its dock (Some docks do not transmit such IR signals).
static volatile int irdIsDetectingIrDock = false;


//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// CAUTION: Items below are FOR DEBUG/TEST ONLY.

// Whether to simulate firmware uploads of IR controller,
//  instead of actually applying them to the IR controller.
static int const irdFwUploadIsSimulated = false;

// Whether to track firmware uploads of IR controller
//  by counting particular kinds of Cresnet packets.
static int const irdFwUploadIsTracked = false;

// Counts "block complete" packets in firmware uploads of IR controller.
static volatile size_t irdFwUploadCountBlock = 0;

// Counts "ready to transfer" packets in firmware uploads of IR controller.
static volatile size_t irdFwUploadCountReady = 0;

//-------------------------------------------------------------------------
// Local function declarations.

static void irdCleanUp (struct IRD_chip *chip);
//static int irdProductHasSupport (void);
static void irdBufferDump (int const condition, unsigned char const *const buffer, size_t const length);
void irdInterruptServiceThreadFunc(struct work_struct *work);
//static void irdOnDockDetection (struct IRD_chip *chip, int const detectedIrFromDock);
//static int irdFwUploadTest (unsigned char const *const pBuf, unsigned short const length);

//static void WaitForAwake(struct work_struct *work);
static void IrdHwDeviceCleanUp (struct IRD_chip *chip);



//-------------------------------------------------------------------------
// Local function definitions.

/**
* \brief    Cleans-up, Prepares for termination/unloading of driver.
* \param    (None).
* \return   (None).
* \author   Arto Kiremitdjian.
* \date     5/1/2012.
*/
void irdCleanUp (struct IRD_chip *chip)
{
    //printk(KERN_DEBUG "+%s.\n", __FUNCTION__);
    //- - - - -
    // NOTE: Not explicitly attempting to terminate any internal threads.
    //       Expect them to end/fail after other resources are cleaned-up.
    //- - - - -
    IrdHwDeviceCleanUp(chip);
    //printk(KERN_DEBUG "-%s.\n", __FUNCTION__);
}

/**
* \brief    Interrupt service task (IST) for messages from IR transceiver.
* \detail   Upon the interrupt event, reads Cresnet packet(s) over I2C bus.
* \note     See Microsoft Visual C++ Help on thread function 'ThreadProc()',
*           because this is an implementation of that kind of function.
* \author   Dennis Griffin, Larry Salant, Arto Kiremitdjian.
* \date     4/3/2012 and earlier.
*/
void irdInterruptServiceThreadFunc(struct work_struct *work)
//unsigned int irdInterruptServiceThreadFunc (void* lpParam)
{
    int result = false; // ASSUME failure.

    //printk(KERN_ERR "RT: +%s.\n", __FUNCTION__);


    // Raise current thread's priority, in order to react to interrupt events
    //  as quickly as possible.  Safe, because the thread mostly blocks/waits.
        {
            //- - - - -
            // CAUTION: Clearing the interrupt before actually handling it,
            //          in order to avoid missing any such interrupts.
            //
            //  If the underlying h/w interrupt occurs again quickly, then
            //  the associated interrupt event should be detected/handled
            //  upon the next iteration of the enclosing wait-loop.
            //- - - - -
        	sysfs_notify(&gChip->IRD_dev->this_device->kobj, NULL, "event");
            enable_irq(gChip->client->irq);

        }
        //...ASSUME that wait-loop ended due to normal termination/clean-up.
        result = 1;

    //printk(KERN_ERR "RT: -%s = int %lu.\n", __FUNCTION__, (unsigned long)(result));
    return;
}



/*************************************************************************
 * ISR function. This function is general, initialized in drivers init
 * function
 ************************************************************************/
static irqreturn_t ird_irq(int irq, void *handle)
{
    struct IRD_chip *chip = (struct IRD_chip *) handle;

    //printk(KERN_ERR "RT: +%s.\n", __FUNCTION__);
    {
    	/* disable further interrupts until this interrupt is processed */
    	disable_irq_nosync(chip->client->irq);
    }

    /* schedule motion signal handling */
    queue_work(ird_interrupt_wq, &chip->Interrupt_work);

    //printk(KERN_ERR "RT: -%s.\n", __FUNCTION__);
    return IRQ_HANDLED;
}


/**
* \brief    Required initialization.
* \detail   Upon success, IR-related h/w is in its inactive/reset state.
* \param    (None).
* \return   intean: Whether this function succeeded.
* \author   Arto Kiremitdjian.
* \date     3/27/2012
*/
int IrdHwDeviceInit (struct IRD_chip *chip)
{

    int result = 0; // ASSUME negative outcome.
    //printk(KERN_ERR "+%s.\n", __FUNCTION__ );

    //- - - - -
    // NOTE: Dummy loop, only to allow early exit with 'break' at any point.
    //       Steps below check for failure, instead of verifying success.
    //       Such negative-logic if-statements can be error-prone, but they
    //       can form a multi-step sequence without nested if-statements.
    //- - - - -
    do {
    	IoPower(chip, 0);
		Reset(chip, 1);

        /* Interrupt setup */
		if (chip->client->irq)
		{
			/* request_irq() will call enable_irq() */
			//printk("RJK: setup IRD INT=0x%x\n", chip->client->irq);
			result = request_irq(chip->client->irq, ird_irq, IRQF_TRIGGER_FALLING, DEVICE_SHORT_NAME, chip);
			if (result)
			{
				printk(KERN_ERR "%s() - ERROR: Could not request IRQ: %d\n", __FUNCTION__, result);
				break;
			}
		}
        // Completed all steps.
        result = true; // Prepare to report success.

    } while(0);

    // Clean-up if any part of the initialization attempt failed.
    if (!result)
    {
    	IrdHwDeviceCleanUp(chip);
    }
    //printk(KERN_ERR "-%s = int %u.\n", __FUNCTION__, (unsigned int)result );
    return result;
}


/**
* \brief    De-initializer.  Performs clean-up.
* \param    (None).
* \return   (None).
* \author   Arto Kiremitdjian.
* \date     3/27/2012
*/
void IrdHwDeviceCleanUp (struct IRD_chip *chip)
{
	//printk(KERN_DEBUG "+%s.\n", __FUNCTION__);

    // NOTE: Typically, clean-up order is opposite of initialization order.

    // Disable the system interrupt and dissociate it from the OS event.
//    hwSignals.SystemInterruptCleanUp();

    // Clean-up the OS event that indicates h/w interrupts.
    //
    //  - NOTE: If an interrupt service thread (IST) is waiting on the event,
    //          then destroying the event will unblock/run the IST due to an
    //          error, rather than a time-out or the signaling of the event.
    //          In other words, within the IST, 'WaitForSingleObject()' will
    //          return a result-code other than 'WAIT_OBJECT_0'.
    //

    // Clean-up the major parts of the IR-related h/w collection.
    hwSignalsCleanUp(chip);

    //printk(KERN_DEBUG "-%s.\n", __FUNCTION__);
}

/**
* \brief    Writes data to the IR transceiver.
* \param    pBuf - Input buffer containing the data to write.
* \param    length - Size of input buffer: Number of bytes to write.
* \return   intean: Whether this function succeeded.
* \author   Arto Kiremitdjian.
* \date     3/21/2012
*/
int IrdHwDeviceWrite (struct IRD_chip *chip, char* pBuf, size_t length)
{
    int result = 0; // ASSUME failure.
    //printk(KERN_ERR "+%s.\n", __FUNCTION__);
    //printk(KERN_ERR "%s: packet %02X %02X %02X %02X %02X %02X\n",
    //    __FUNCTION__,
    //    ((0 < length) ? (unsigned int)(pBuf[0]) : 0),
    //    ((1 < length) ? (unsigned int)(pBuf[1]) : 0),
    //    ((2 < length) ? (unsigned int)(pBuf[2]) : 0),
    //    ((3 < length) ? (unsigned int)(pBuf[3]) : 0),
    //    ((4 < length) ? (unsigned int)(pBuf[4]) : 0),
    //    ((5 < length) ? (unsigned int)(pBuf[5]) : 0));


    result = IrdI2cInterfaceWrite(chip->client, pBuf, length);

    //printk(KERN_ERR "-%s = int %u.\n", __FUNCTION__, (unsigned int)result );
    return result;
}

/**
* \brief    Reads data from the IR transceiver.
* \detail   Usage: Only an interrupt service thread (IST) should invoke the
*           function, in order to handle/process the interrupt that indicates
*           that the IR transceiver has an outbound (Cresnet) message/packet.
* \param    pBuf - Output buffer for whatever data have been read.
* \param    length - Size of output buffer: Number of bytes.
* \return   intean: Whether this function succeeded.
* \author   Arto Kiremitdjian.
* \date     3/27/2012
*/
int IrdHwDeviceRead (struct IRD_chip *chip, char* pBuf, size_t length)
{
    // Delegate to a sub-object.
	int res;

	res = IrdI2cInterfaceRead(chip->client, pBuf, length);

    //printk(KERN_ERR "%s: packet %02X %02X %02X %02X %02X %02X\n",
    //    __FUNCTION__,
    //    ((0 < length) ? (unsigned int)(pBuf[0]) : 0),
    //    ((1 < length) ? (unsigned int)(pBuf[1]) : 0),
    //    ((2 < length) ? (unsigned int)(pBuf[2]) : 0),
    //    ((3 < length) ? (unsigned int)(pBuf[3]) : 0),
    //    ((4 < length) ? (unsigned int)(pBuf[4]) : 0),
    //    ((5 < length) ? (unsigned int)(pBuf[5]) : 0));

    return res;
}

//-------------------------------------------------------------------------
// Global function definitions.


/**************************************************************************
* \brief    See Microsoft Visual C++ Help on stream driver 'XXX_Read()',
*           because this is an implementation of that kind of function.
* \author   Larry Salant, Arto Kiremitdjian.
* \date     12/19/2012
*/
static ssize_t IRD_Read(struct file *file, char __user *buf,
			  size_t count, loff_t *ppos)
{
    //- - - - -
    // TODO: Handle failed message-queue read on '..._Close()'/'..._Deinit()':
    //  On entry to this function, remember the device context associated with
    //  the given open context.  Later, if the message-queue read operation
    //  fails, check whether the session is closed or the device is
    //  de-initialized.  If so, report that zero bytes were read, instead of
    //  reporting an (unexpected) failure.
    //- - - - -

    int result = -1; // ASSUME failure.
    char* buffer;

    //printk(KERN_ERR "+%s([context], %08lXh, %lu).\n",
    //          __FUNCTION__,
    //          (unsigned long)(buf),
    //          (unsigned long)(count));

    if (count > ird_buf_len) {
		void *p;
		result = -ENOMEM;
		p = kmalloc(count, GFP_KERNEL);
		if (!p)
			return -ENOMEM;
		if(ird_buf)
			kfree(ird_buf);
		ird_buf = p;
		ird_buf_len = count;
	}

    buffer = ird_buf;



    //- - - - -
    // NOTE: Not using the same thread-synchronization scheme as the other
    //       entry-point functions of the device driver, as in the statement
    //       'mutex_lock(&chip->fs_mutex);'.  Reasons:
    //
    //       - Code below might wait/block "forever" on a message queue.
    //         If it did so after getting ownership of the synchronization
    //         object, then all other operations would be blocked with it.
    //
    //       - Code below does not access the shared resources that are
    //         protected/synchronized by the other entry-point functions.
    //- - - - -

    //printk(KERN_ERR "%s: Received interrupt %x",__FUNCTION__, gChip);
    result = IrdHwDeviceRead(gChip, buffer, count);
    if (result > 0)
    {
    	//printk(KERN_ERR "%s: Packet header %02X %02X.\n", __FUNCTION__, (*buffer),*(buffer+1));
    	if (copy_to_user(buf, buffer, count))
			result = -EFAULT;
    }
    else
	{
		printk(KERN_ERR "%s: FAILED to read packet after interrupt.\n", __FUNCTION__ );
	}
    return result;
}

/**************************************************************************
* \brief    See Microsoft Visual C++ Help on stream driver 'XXX_Write()',
*           because this is an implementation of that kind of function.
* \author   Larry Salant, Arto Kiremitdjian.
* \date     12/19/2012
*/
static ssize_t IRD_Write(struct file *file, const char __user *buf,
			   size_t count, loff_t *ppos)

{
    unsigned int result = (unsigned int)(-1); // ASSUME failure.
    struct IRD_chip *chip = gChip;

    //printk(KERN_ERR "+%s([context], %08lXh, %lu).\n",
    //          __FUNCTION__,
    //          (unsigned long)(buf),
    //          (unsigned long)(count));

    if (count > ird_buf_len) {
		void *p;
		result = -ENOMEM;
		p = kmalloc(count, GFP_KERNEL);
		if (!p)
			return -ENOMEM;
		if(ird_buf)
			kfree(ird_buf);
		ird_buf = p;
		ird_buf_len = count;
	}

    if (copy_from_user(ird_buf, buf, count)) {
		kfree(ird_buf);
		return -EFAULT;
	}
    // Verify parameter(s).
    //  - Such verification includes range-checking parameter values that
    //    will be cast to other types when passed to callees further below.
    if ((NULL != buf) &&
        ((unsigned short)(-1) >= count))
    {
        mutex_lock(&chip->fs_mutex);

        // Alias for input buffer: Matches desired data-type, Convenient.
		result = IrdHwDeviceWrite(chip, ird_buf, (unsigned short)(count));

        mutex_unlock(&chip->fs_mutex);
    }
    printk(KERN_DEBUG "-%s = %lu bytes.\n", __FUNCTION__, (unsigned long)(result));
    return result;
}

const struct file_operations IRD_fops = {
	.owner		= THIS_MODULE,
//	.llseek		= IRD_llseek,
	.read		= IRD_Read,
	.write		= IRD_Write,
//	.unlocked_ioctl	= IRD_unlocked_ioctl,
};

static ssize_t ioPower_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct IRD_chip *chip = dev_get_drvdata(dev);

	int value;
	sscanf(buf,"%u",&value);

	IoPower(chip, value);
	return count;
}

static DEVICE_ATTR(ioPower, S_IWUSR, NULL, ioPower_store);

static ssize_t reset_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct IRD_chip *chip = dev_get_drvdata(dev);

	int value;
	sscanf(buf,"%u",&value);

	Reset(chip, value);
	return count;
}

static DEVICE_ATTR(reset, S_IWUSR, NULL, reset_store);

static ssize_t ird_event(struct device *dev,
				       struct device_attribute *attr, char *buf)
{
	return 0;
}

static DEVICE_ATTR(event, S_IRUSR, ird_event, NULL);

static struct attribute *ird_attributes[] = {
	&dev_attr_ioPower.attr,
	&dev_attr_reset.attr,
	&dev_attr_event.attr,
	NULL
};

static struct attribute_group ird_attribute_group = {
	.attrs = ird_attributes
};

static struct miscdevice IRD_dev = {
	.minor 		= MISC_DYNAMIC_MINOR,
	.name  		= DEVICE_SHORT_NAME,
//	.nodename	= "pds",
	.fops  		= &IRD_fops
};

/**************************************************************************
* \brief    See Microsoft Visual C++ Help on stream driver 'XXX_Init()',
*           because this is an implementation of that kind of function.
* \author   Dennis Griffin, Larry Salant, Arto Kiremitdjian.mxc_i2c0_600
* \date     11/14/2011
*/
static int __devinit IRD_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{
	struct IRD_chip *chip;
	struct IRD_platform_data *pdata;
	int ret=0;
	int result = 0; // ASSUME negative outcome.

	//printk(KERN_ERR "RT: +%s.\n", __FUNCTION__);

	gChip = chip = kzalloc(sizeof(struct IRD_chip), GFP_KERNEL);
	if (!chip) {
		dev_err(&client->dev, "failed to allocate driver data\n");
		goto err_out;
	}

	//printk(KERN_ERR "RT: +%s gChip %x\n", __FUNCTION__, gChip);
	chip->client = client;

	pdata = chip->pdata = client->dev.platform_data;
	i2c_set_clientdata(client, chip);

	dev_set_drvdata(IRD_dev.this_device, chip);
	chip->IRD_dev = &IRD_dev;

	if(chip->pdata->reset.gpio_pin)
		gpio_request(chip->pdata->reset.gpio_pin, "ir_rst_n");
	if(chip->pdata->power.gpio_pin)
		gpio_request(chip->pdata->power.gpio_pin, "ir_pwr_ctrl");

    mutex_init(&chip->fs_mutex);

	mutex_lock(&chip->fs_mutex);

    // NOTE: Does not actually loop.  Only allows early exit upon failure.
    do {
        //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // NOTE: Each step below performs an initialization operation and
        //       checks for its failure, instead of its success.  Although
        //       such negative-logic if-statements can be error-prone, they
        //       are easier to read/understand in a long sequence of steps.

        //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // Verify that the current product supports an IR dock.

        //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // Initialize the collection of IR-related h/w.

        if (!IrdHwDeviceInit(chip))
        {
            printk(KERN_ERR "%s: FAILED to initialize h/w.\r\n", __FUNCTION__);
            goto err_out; // Failed: Exit enclosing pseudo-loop early.
        }
        //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // Start the interrupt service thread (IST).

        /* Prepare our worker structure prior to setting up the timer/ISR */
		INIT_WORK(&chip->Interrupt_work, irdInterruptServiceThreadFunc);
//		INIT_WORK(&chip->WakeUp_work, PerformWakeUp);

//		atomic_set(&chip->irq_enabled, 1);

		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        //...All steps above passed: Prepare to report success.

        result = 1; // TODO: Return a device context, instead.

    } while(false);

    //printk(KERN_ERR "RT: +%s 3.\n", __FUNCTION__);

    mutex_unlock(&chip->fs_mutex);

    ret = sysfs_create_group(&IRD_dev.this_device->kobj, &ird_attribute_group);
	if (ret)
		goto err_out;

    printk(KERN_DEBUG "-%s = %08lXh.\n", __FUNCTION__, (unsigned long)(result));
	//printk(KERN_ERR "RT: -%s.\n", __FUNCTION__);
    return 0;

err_out:
printk(KERN_ERR "%s() - ERROR: Exiting probe without completion.\n", __FUNCTION__);
	irdCleanUp(chip);
	mutex_unlock(&chip->fs_mutex);
	kfree(chip);
	i2c_set_clientdata(client, NULL);
	return ret;
}

/**************************************************************************
* \brief    See Microsoft Visual C++ Help on stream driver 'XXX_Deinit()',
*           because this is an implementation of that kind of function.
* \author   Dennis Griffin, Arto Kiremitdjian.
* \date     11/14/2011
*/
static int __devexit IRD_remove(struct i2c_client *client)
{
//	struct i2c_client *client = to_i2c_client(dev);
	struct IRD_chip *chip = i2c_get_clientdata(client);

    //printk(KERN_ERR "=%s.\n", __FUNCTION__ );

    mutex_lock(&chip->fs_mutex);
    irdCleanUp(chip);
    mutex_unlock(&chip->fs_mutex);
    mutex_destroy(&chip->fs_mutex);
    return 1;
}

/**************************************************************************
* \brief    See Microsoft Visual C++ Help on stream driver 'XXX_PowerUp()',
*           because this is an implementation of that kind of function.
* \author   Arto Kiremitdjian.
* \date     4/4/2012.
*/
static int IRD_PowerUp (struct i2c_client *client)
{
//	struct i2c_client *client = to_i2c_client(dev);
//	struct IRD_chip *chip = i2c_get_clientdata(client);

    //printk(KERN_DEBUG "+%s.\n", __FUNCTION__ );

	// Removed code reading system power status.
	// Trying to read system power status in PowerUp is not a good idea.
	// On resume, UIHAL PM will determine the AC line status and let IRD driver know.

    //printk(KERN_DEBUG "-%s.\n", __FUNCTION__ );

    return 0;
}

/**************************************************************************
* \brief    See Microsoft Visual C++ Help on stream driver 'XXX_PowerDown()',
*           because this is an implementation of that kind of function.
* \author   Arto Kiremitdjian.
* \date     4/4/2012.
*/
static int IRD_PowerDown (struct i2c_client *client, pm_message_t mesg)
{
//	struct i2c_client *client = to_i2c_client(dev);
	struct IRD_chip *chip = i2c_get_clientdata(client);

    //printk(KERN_DEBUG "+%s.\n", __FUNCTION__ );

    mutex_lock(&chip->fs_mutex);

    // Nothing to do: Positive outcome.

    mutex_unlock(&chip->fs_mutex);
    //printk(KERN_DEBUG "-%s.\n", __FUNCTION__ );

    return 0;
}

static const struct i2c_device_id IRD_ids[] = {
	{ "IRD", 0 },
	{ }
};

MODULE_DEVICE_TABLE(i2c, IRD_ids);

static struct i2c_driver IRD_i2c_driver = {
	.driver	= {
		.name	= "IRD",
	},
	.probe		= IRD_probe,
	.remove		= __devexit_p(IRD_remove),
	.suspend	= IRD_PowerDown,
	.resume		= IRD_PowerUp,
	.id_table	= IRD_ids,
};

static int __init IRD_init(void)
{
	int ret;
	//printk(KERN_ERR "RT: +%s.\n", __FUNCTION__);
	ret = misc_register(&IRD_dev);

	if (ret != 0)
	{
		printk("failed to register IRD device %d\n",ret);
		goto out;
	}

	ird_interrupt_wq = create_singlethread_workqueue("ird_interrupt_wq");
	ird_wakeup_wq = create_singlethread_workqueue("ird_wakeup_wq");

	ret = i2c_add_driver(&IRD_i2c_driver);
out:
	//printk(KERN_ERR "RT: -%s.\n", __FUNCTION__);
	return ret;
}
module_init(IRD_init);

static void __exit IRD_exit(void)
{
	i2c_del_driver(&IRD_i2c_driver);
}
module_exit(IRD_exit);

MODULE_AUTHOR("Crestron Electronics");
MODULE_DESCRIPTION("IRD Dock driver");
MODULE_LICENSE("GPL v2");
